import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Leaf, Shield, Award, Clock, ExternalLink, Code } from "lucide-react";
import { useQuery } from "@tanstack/react-query";

export default function Home() {
  const [showEmbedCode, setShowEmbedCode] = useState(false);
  
  const { data: embedData } = useQuery<{iframe: string, script: string}>({
    queryKey: ["/api/widget/embed"],
    enabled: showEmbedCode
  });

  return (
    <div className="min-h-screen bg-eco-bg">
      {/* Header */}
      <header className="bg-white border-b border-eco-light">
        <div className="max-w-7xl mx-auto px-4 py-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <div className="w-12 h-12 bg-eco-primary rounded-xl flex items-center justify-center">
                <Leaf className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold text-eco-secondary">Green Tech Shredding</h1>
                <p className="text-sm text-gray-600">Secure Document Destruction</p>
              </div>
            </div>
            <div className="flex items-center space-x-6">
              <Badge variant="outline" className="border-eco-primary text-eco-primary">
                <Shield className="w-3 h-3 mr-1" />
                HIPAA Compliant
              </Badge>
              <Badge variant="outline" className="border-eco-primary text-eco-primary">
                <Award className="w-3 h-3 mr-1" />
                Certified Destruction
              </Badge>
            </div>
          </div>
        </div>
      </header>

      {/* Hero Section */}
      <section className="py-16">
        <div className="max-w-4xl mx-auto px-4 text-center">
          <h2 className="text-4xl font-bold text-eco-secondary mb-6">
            Professional Document Shredding Services
          </h2>
          <p className="text-xl text-gray-600 mb-8">
            Get an instant quote for secure, certified document destruction. 
            Our lead generation wizard makes it easy for customers to request shredding services.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-eco-primary hover:bg-green-600" asChild>
              <a href="/widget">
                <ExternalLink className="w-4 h-4 mr-2" />
                Try Live Widget
              </a>
            </Button>
            <Button 
              size="lg" 
              variant="outline" 
              className="border-eco-primary text-eco-primary hover:bg-eco-light"
              onClick={() => setShowEmbedCode(!showEmbedCode)}
            >
              <Code className="w-4 h-4 mr-2" />
              Get Embed Code
            </Button>
          </div>
        </div>
      </section>

      {/* Embed Code Section */}
      {showEmbedCode && embedData && (
        <section className="py-8 bg-white">
          <div className="max-w-4xl mx-auto px-4">
            <h3 className="text-2xl font-bold text-eco-secondary mb-6 text-center">
              Embed This Widget
            </h3>
            <div className="grid md:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">iframe Embed</CardTitle>
                </CardHeader>
                <CardContent>
                  <textarea
                    readOnly
                    value={embedData?.iframe || ''}
                    className="w-full h-24 p-3 text-sm font-mono bg-gray-50 border rounded-lg"
                    onClick={(e) => e.currentTarget.select()}
                  />
                </CardContent>
              </Card>
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">JavaScript Embed</CardTitle>
                </CardHeader>
                <CardContent>
                  <textarea
                    readOnly
                    value={embedData?.script || ''}
                    className="w-full h-24 p-3 text-sm font-mono bg-gray-50 border rounded-lg"
                    onClick={(e) => e.currentTarget.select()}
                  />
                </CardContent>
              </Card>
            </div>
          </div>
        </section>
      )}

      {/* Features */}
      <section className="py-16 bg-white">
        <div className="max-w-6xl mx-auto px-4">
          <h3 className="text-3xl font-bold text-eco-secondary text-center mb-12">
            Why Choose Our Lead Generation Wizard?
          </h3>
          <div className="grid md:grid-cols-3 gap-8">
            <Card className="border-eco-light">
              <CardContent className="p-6 text-center">
                <div className="w-16 h-16 bg-eco-light rounded-xl flex items-center justify-center mx-auto mb-4">
                  <Clock className="w-8 h-8 text-eco-primary" />
                </div>
                <h4 className="text-xl font-semibold text-eco-secondary mb-2">Instant Quotes</h4>
                <p className="text-gray-600">
                  Real-time pricing based on volume and service type. 
                  Customers get accurate estimates immediately.
                </p>
              </CardContent>
            </Card>

            <Card className="border-eco-light">
              <CardContent className="p-6 text-center">
                <div className="w-16 h-16 bg-eco-light rounded-xl flex items-center justify-center mx-auto mb-4">
                  <Shield className="w-8 h-8 text-eco-primary" />
                </div>
                <h4 className="text-xl font-semibold text-eco-secondary mb-2">CRM Integration</h4>
                <p className="text-gray-600">
                  Automatic lead submission to PerfexCRM with email notifications 
                  for seamless follow-up.
                </p>
              </CardContent>
            </Card>

            <Card className="border-eco-light">
              <CardContent className="p-6 text-center">
                <div className="w-16 h-16 bg-eco-light rounded-xl flex items-center justify-center mx-auto mb-4">
                  <Code className="w-8 h-8 text-eco-primary" />
                </div>
                <h4 className="text-xl font-semibold text-eco-secondary mb-2">Easy Embedding</h4>
                <p className="text-gray-600">
                  Simple iframe or JavaScript integration. 
                  Works on any website with responsive design.
                </p>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-eco-secondary text-white py-8">
        <div className="max-w-6xl mx-auto px-4 text-center">
          <div className="flex items-center justify-center space-x-3 mb-4">
            <div className="w-8 h-8 bg-eco-primary rounded-lg flex items-center justify-center">
              <Leaf className="w-4 h-4 text-white" />
            </div>
            <span className="text-xl font-semibold">Green Tech Shredding</span>
          </div>
          <p className="text-gray-300 mb-4">
            Professional document destruction services with environmental responsibility
          </p>
          <div className="flex items-center justify-center space-x-6 text-sm">
            <span>📞 (800) 555-SHRED</span>
            <span>✉️ cnngraphics@gmail.com</span>
            <span>🕒 Available 24/7</span>
          </div>
        </div>
      </footer>
    </div>
  );
}
