import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Box, Archive, ShoppingBag, Package } from "lucide-react";
import { WizardData } from "@/lib/types";
import { calculateTotalBoxes, calculateTotalWeight } from "@/lib/quote-calculator";

interface VolumeCalculatorProps {
  data: WizardData;
  onUpdate: (data: Partial<WizardData>) => void;
  onNext: () => void;
  onBack: () => void;
}

const volumeTypes = [
  {
    key: "fileBoxes",
    title: "File Boxes",
    description: "15\" × 12\" × 10\" standard boxes",
    icon: Box,
    color: "bg-blue-100 text-blue-600"
  },
  {
    key: "filingCabinets",
    title: "Filing Cabinets",
    description: "4-drawer standard cabinets",
    icon: Archive,
    color: "bg-purple-100 text-purple-600"
  },
  {
    key: "bags",
    title: "Bags/Containers",
    description: "Large garbage bags or containers",
    icon: ShoppingBag,
    color: "bg-green-100 text-green-600"
  },
  {
    key: "pallets",
    title: "Pallets",
    description: "Standard 40\" × 48\" pallets",
    icon: Package,
    color: "bg-orange-100 text-orange-600"
  }
];

export function VolumeCalculator({ data, onUpdate, onNext, onBack }: VolumeCalculatorProps) {
  const volumeData = data.volumeData || { fileBoxes: 0, filingCabinets: 0, bags: 0, pallets: 0 };
  const totalBoxes = calculateTotalBoxes(volumeData);
  const totalWeight = calculateTotalWeight(totalBoxes);

  const updateVolume = (key: string, value: number) => {
    const newVolumeData = { ...volumeData, [key]: Math.max(0, value) };
    onUpdate({ volumeData: newVolumeData });
  };

  const incrementVolume = (key: string) => {
    updateVolume(key, (volumeData[key as keyof typeof volumeData] || 0) + 1);
  };

  const decrementVolume = (key: string) => {
    updateVolume(key, (volumeData[key as keyof typeof volumeData] || 0) - 1);
  };

  return (
    <div className="flex flex-col h-full">
      <div className="flex-1 p-8 pb-24 md:pb-8">
        <div className="text-center mb-8">
          <h3 className="text-2xl font-bold text-eco-secondary mb-2">
            How much material do you have?
          </h3>
          <p className="text-gray-600">Use our calculator to estimate your volume accurately</p>
        </div>

        <div className="max-w-3xl mx-auto">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
            {volumeTypes.map((type) => {
              const Icon = type.icon;
              const value = volumeData[type.key as keyof typeof volumeData] || 0;
              
              return (
                <div key={type.key} className="bg-gray-50 rounded-xl p-6">
                  <div className="flex items-center space-x-3 mb-4">
                    <div className={`w-10 h-10 ${type.color} rounded-lg flex items-center justify-center`}>
                      <Icon className="w-5 h-5" />
                    </div>
                    <div>
                      <h4 className="font-semibold text-gray-900">{type.title}</h4>
                      <p className="text-sm text-gray-600">{type.description}</p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-3">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => decrementVolume(type.key)}
                      className="w-10 h-10 p-0 border-2 border-gray-400 hover:border-eco-primary text-[14pt]"
                      disabled={value <= 0}
                      data-testid={`button-decrement-${type.key}`}
                    >
                      -
                    </Button>
                    <Input
                      type="number"
                      value={value}
                      onChange={(e) => updateVolume(type.key, parseInt(e.target.value) || 0)}
                      className="flex-1 text-center text-lg font-semibold border-2 border-gray-400 focus:border-eco-primary"
                      min="0"
                      data-testid={`input-${type.key}`}
                    />
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => incrementVolume(type.key)}
                      className="w-10 h-10 p-0 border-2 border-gray-400 hover:border-eco-primary text-[14pt]"
                      data-testid={`button-increment-${type.key}`}
                    >
                      +
                    </Button>
                  </div>
                </div>
              );
            })}
          </div>

          {/* Volume Summary */}
          <div className="bg-eco-light rounded-xl p-6 mb-6">
            <h4 className="font-semibold text-eco-secondary mb-4">Volume Summary</h4>
            <div className="grid grid-cols-2 gap-4">
              <div className="text-center">
                <div className="text-2xl font-bold text-eco-secondary">{totalBoxes}</div>
                <div className="text-sm text-gray-600">Equivalent Boxes</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold text-eco-secondary">{totalWeight}</div>
                <div className="text-sm text-gray-600">Estimated Pounds</div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Desktop Buttons */}
      <div className="hidden md:flex justify-between p-8 pt-0">
        <Button 
          onClick={onBack}
          variant="outline"
          className="px-6 py-3"
        >
          ← Back
        </Button>
        <Button 
          onClick={onNext}
          disabled={totalBoxes === 0}
          className="bg-eco-primary text-white px-8 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
        >
          Continue to Service Options
          <span className="ml-2">→</span>
        </Button>
      </div>

      {/* Mobile Sticky Buttons */}
      <div className="md:hidden fixed bottom-0 left-0 right-0 p-4 bg-white border-t border-gray-200 shadow-lg">
        <div className="flex gap-3">
          <Button 
            onClick={onBack}
            variant="outline"
            className="flex-1 py-4"
          >
            ← Back
          </Button>
          <Button 
            onClick={onNext}
            disabled={totalBoxes === 0}
            className="flex-2 bg-eco-primary text-white py-4 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
          >
            Continue
            <span className="ml-2">→</span>
          </Button>
        </div>
      </div>
    </div>
  );
}
