import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Truck, Building, MapPin, Check, X } from "lucide-react";
import { WizardData } from "@/lib/types";
import { calculateServiceQuote } from "@/lib/quote-calculator";

interface ServiceSelectionProps {
  data: WizardData;
  onUpdate: (data: Partial<WizardData>) => void;
  onNext: () => void;
  onBack: () => void;
}

export function ServiceSelection({ data, onUpdate, onNext, onBack }: ServiceSelectionProps) {
  const volumeData = data.volumeData || { fileBoxes: 0, filingCabinets: 0, bags: 0, pallets: 0 };
  const selectedService = data.serviceType;

  const services = [
    {
      id: "mobile",
      title: "Mobile Shredding",
      description: "We come to you with our secure truck",
      icon: Truck,
      color: "bg-blue-100 text-blue-600",
      features: [
        "Witnessed destruction",
        "Certificate of destruction",
        "Highest security level",
        "Same-day scheduling"
      ],
      quote: calculateServiceQuote("mobile", volumeData),
      recommended: true
    },
    {
      id: "offsite",
      title: "Off-Site Shredding",
      description: "Pickup and secure facility destruction",
      icon: Building,
      color: "bg-green-100 text-green-600",
      features: [
        "Secure pickup service",
        "Certificate of destruction",
        "Competitive pricing",
        "Large volume friendly"
      ],
      quote: calculateServiceQuote("offsite", volumeData)
    },
    {
      id: "dropoff",
      title: "Drop-Off",
      description: "Bring materials to our facility",
      icon: MapPin,
      color: "bg-purple-100 text-purple-600",
      features: [
        "Most economical option",
        "Immediate service",
        "Convenient locations"
      ],
      limitations: ["No certificate provided"],
      quote: calculateServiceQuote("dropoff", volumeData)
    }
  ];

  const selectService = (serviceId: string) => {
    onUpdate({ serviceType: serviceId });
  };

  return (
    <div className="flex flex-col h-full">
      <div className="flex-1 p-8 pb-24 md:pb-8">
        <div className="text-center mb-8">
          <h3 className="text-2xl font-bold text-eco-secondary mb-2">Choose your service type</h3>
          <p className="text-gray-600">We recommend the best option based on your volume</p>
        </div>

        <div className="max-w-4xl mx-auto">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {services.map((service) => {
              const Icon = service.icon;
              const isSelected = selectedService === service.id;
              
              return (
                <button
                  key={service.id}
                  onClick={() => selectService(service.id)}
                  className={`border-2 rounded-xl p-6 hover:border-eco-primary hover:shadow-lg transition-all duration-300 text-left relative ${
                    isSelected 
                      ? 'border-eco-primary bg-eco-light' 
                      : 'border-gray-200'
                  }`}
                >
                  {service.recommended && (
                    <Badge className="absolute top-4 right-4 bg-eco-accent text-white">
                      Recommended
                    </Badge>
                  )}
                  
                  <div className="text-center mb-6">
                    <div className={`w-16 h-16 ${service.color} rounded-xl flex items-center justify-center mx-auto mb-4`}>
                      <Icon className="text-2xl w-8 h-8" />
                    </div>
                    <h4 className="text-xl font-bold text-gray-900 mb-2">{service.title}</h4>
                    <p className="text-gray-600 text-sm">{service.description}</p>
                  </div>
                  
                  <div className="space-y-3 mb-6">
                    {service.features.map((feature) => (
                      <div key={feature} className="flex items-center text-sm">
                        <Check className="text-eco-primary mr-2 w-4 h-4" />
                        <span>{feature}</span>
                      </div>
                    ))}
                    {service.limitations?.map((limitation) => (
                      <div key={limitation} className="flex items-center text-sm">
                        <X className="text-red-400 mr-2 w-4 h-4" />
                        <span className="text-gray-400">{limitation}</span>
                      </div>
                    ))}
                  </div>

                  <div className="border-t pt-4">
                    <div className="text-center">
                      <div className="text-2xl font-bold text-eco-secondary">{service.quote}</div>
                      <div className="text-sm text-gray-600">
                        {service.id === "dropoff" ? "per pound" : "for 1-10 boxes"}
                      </div>
                    </div>
                  </div>
                </button>
              );
            })}
          </div>
        </div>
      </div>

      {/* Desktop Buttons */}
      <div className="hidden md:flex justify-between p-8 pt-0">
        <Button 
          onClick={onBack}
          variant="outline"
          className="px-6 py-3"
        >
          ← Back
        </Button>
        <Button 
          onClick={onNext}
          disabled={!selectedService}
          className="bg-eco-primary text-white px-8 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
        >
          Continue to Location
          <span className="ml-2">→</span>
        </Button>
      </div>

      {/* Mobile Sticky Buttons */}
      <div className="md:hidden fixed bottom-0 left-0 right-0 p-4 bg-white border-t border-gray-200 shadow-lg">
        <div className="flex gap-3">
          <Button 
            onClick={onBack}
            variant="outline"
            className="flex-1 py-4"
          >
            ← Back
          </Button>
          <Button 
            onClick={onNext}
            disabled={!selectedService}
            className="flex-2 bg-eco-primary text-white py-4 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
          >
            Continue
            <span className="ml-2">→</span>
          </Button>
        </div>
      </div>
    </div>
  );
}
