import { Button } from "@/components/ui/button";
import { FileText, HardDrive, Laptop, Archive } from "lucide-react";
import { WizardData } from "@/lib/types";

interface MaterialSelectionProps {
  data: WizardData;
  onUpdate: (data: Partial<WizardData>) => void;
  onNext: () => void;
}

const materialTypes = [
  {
    id: "documents",
    title: "Paper Documents",
    description: "Files, reports, contracts, medical records",
    icon: FileText,
    color: "bg-blue-100 text-blue-600",
    hoverColor: "group-hover:bg-eco-primary group-hover:text-white",
    badge: "Most Common",
    badgeColor: "text-eco-secondary"
  },
  {
    id: "hard-drives",
    title: "Hard Drives",
    description: "Computer drives, SSDs, storage devices",
    icon: HardDrive,
    color: "bg-purple-100 text-purple-600",
    hoverColor: "group-hover:bg-eco-primary group-hover:text-white",
    badge: "High Security",
    badgeColor: "text-amber-600"
  },
  {
    id: "electronics",
    title: "Electronics",
    description: "Phones, tablets, computers, printers",
    icon: Laptop,
    color: "bg-green-100 text-green-600",
    hoverColor: "group-hover:bg-eco-primary group-hover:text-white",
    badge: "Eco-Friendly",
    badgeColor: "text-eco-primary"
  },
  {
    id: "mixed",
    title: "Mixed Materials",
    description: "Combination of documents and electronics",
    icon: Archive,
    color: "bg-orange-100 text-orange-600",
    hoverColor: "group-hover:bg-eco-primary group-hover:text-white",
    badge: "Custom Solution",
    badgeColor: "text-gray-500"
  }
];

export function MaterialSelection({ data, onUpdate, onNext }: MaterialSelectionProps) {
  const selectedTypes = data.materialTypes || [];

  const toggleMaterialType = (typeId: string) => {
    const newTypes = selectedTypes.includes(typeId)
      ? selectedTypes.filter(type => type !== typeId)
      : [...selectedTypes, typeId];
    
    onUpdate({ materialTypes: newTypes });
  };

  const isSelected = (typeId: string) => selectedTypes.includes(typeId);

  return (
    <div className="flex flex-col h-full">
      <div className="flex-1 p-8 pb-24 md:pb-8">
        <div className="text-center mb-8">
          <h3 className="text-2xl font-bold text-eco-secondary mb-2">
            What type of material needs shredding?
          </h3>
          <p className="text-gray-600">Select all that apply to get the most accurate quote</p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 max-w-2xl mx-auto">
          {materialTypes.map((type) => {
            const Icon = type.icon;
            const selected = isSelected(type.id);
            
            return (
              <button
                key={type.id}
                onClick={() => toggleMaterialType(type.id)}
                className={`p-6 border-2 rounded-xl hover:border-eco-primary hover:bg-eco-light transition-all duration-300 text-left group ${
                  selected 
                    ? 'border-eco-primary bg-eco-light' 
                    : 'border-gray-200'
                }`}
              >
                <div className="flex items-start space-x-4">
                  <div className={`w-12 h-12 ${type.color} rounded-lg flex items-center justify-center transition-colors ${type.hoverColor}`}>
                    <Icon className="text-xl w-6 h-6" />
                  </div>
                  <div className="flex-1">
                    <h4 className="font-semibold text-gray-900 mb-1">{type.title}</h4>
                    <p className="text-sm text-gray-600">{type.description}</p>
                    <div className={`mt-2 text-xs font-medium ${type.badgeColor}`}>
                      {type.badge}
                    </div>
                  </div>
                </div>
              </button>
            );
          })}
        </div>
      </div>

      {/* Desktop Button */}
      <div className="hidden md:flex justify-center p-8 pt-0">
        <Button 
          onClick={onNext}
          disabled={selectedTypes.length === 0}
          className="bg-eco-primary text-white px-8 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
        >
          Continue to Volume Calculator
          <span className="ml-2">→</span>
        </Button>
      </div>

      {/* Mobile Sticky Button */}
      <div className="md:hidden fixed bottom-0 left-0 right-0 p-4 bg-white border-t border-gray-200 shadow-lg">
        <Button 
          onClick={onNext}
          disabled={selectedTypes.length === 0}
          className="w-full bg-eco-primary text-white py-4 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
        >
          Continue to Volume Calculator
          <span className="ml-2">→</span>
        </Button>
      </div>
    </div>
  );
}
