import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { Zap, Calendar, CalendarDays } from "lucide-react";
import { WizardData } from "@/lib/types";

interface LocationTimelineProps {
  data: WizardData;
  onUpdate: (data: Partial<WizardData>) => void;
  onNext: () => void;
  onBack: () => void;
}

const timelineOptions = [
  {
    id: "asap",
    title: "ASAP",
    description: "Within 24-48 hours",
    icon: Zap,
    color: "text-eco-accent"
  },
  {
    id: "week",
    title: "This Week",
    description: "Within 7 days",
    icon: Calendar,
    color: "text-eco-primary"
  },
  {
    id: "flexible",
    title: "Flexible",
    description: "Schedule later",
    icon: CalendarDays,
    color: "text-gray-500"
  }
];

const specialRequirements = [
  { id: "hipaa", label: "HIPAA Compliance Required" },
  { id: "witnessed", label: "Witnessed Destruction Required" },
  { id: "recurring", label: "Recurring Service Needed" },
  { id: "large-volume", label: "Large Volume (>100 boxes)" }
];

export function LocationTimeline({ data, onUpdate, onNext, onBack }: LocationTimelineProps) {
  const zipCode = data.zipCode || "";
  const timeline = data.timeline || "";
  const requirements = data.specialRequirements || [];

  const updateZipCode = (value: string) => {
    onUpdate({ zipCode: value });
  };

  const selectTimeline = (timelineId: string) => {
    onUpdate({ timeline: timelineId });
  };

  const toggleRequirement = (requirementId: string) => {
    const newRequirements = requirements.includes(requirementId)
      ? requirements.filter(req => req !== requirementId)
      : [...requirements, requirementId];
    
    onUpdate({ specialRequirements: newRequirements });
  };

  const isValid = zipCode.length >= 5 && timeline;

  return (
    <div className="flex flex-col h-full">
      <div className="flex-1 p-8 pb-24 md:pb-8">
        <div className="text-center mb-8">
          <h3 className="text-2xl font-bold text-eco-secondary mb-2">Location and scheduling</h3>
          <p className="text-gray-600">Help us provide accurate pricing and availability</p>
        </div>

        <div className="max-w-2xl mx-auto space-y-6">
          {/* Location */}
          <div>
            <Label className="block text-sm font-semibold text-gray-700 mb-2">
              Service Location
            </Label>
            <Input
              type="text"
              placeholder="Enter ZIP code"
              value={zipCode}
              onChange={(e) => updateZipCode(e.target.value)}
              className="w-full"
            />
            <p className="text-sm text-gray-500 mt-1">
              We'll find the closest service provider in your area
            </p>
          </div>

          {/* Timeline */}
          <div>
            <Label className="block text-sm font-semibold text-gray-700 mb-2">
              When do you need service?
            </Label>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
              {timelineOptions.map((option) => {
                const Icon = option.icon;
                const isSelected = timeline === option.id;
                
                return (
                  <button
                    key={option.id}
                    onClick={() => selectTimeline(option.id)}
                    className={`p-4 border-2 rounded-lg hover:border-eco-primary hover:bg-eco-light transition-all duration-300 ${
                      isSelected 
                        ? 'border-eco-primary bg-eco-light' 
                        : 'border-gray-200'
                    }`}
                  >
                    <div className="text-center">
                      <Icon className={`${option.color} text-xl mb-2 w-6 h-6 mx-auto`} />
                      <div className="font-semibold">{option.title}</div>
                      <div className="text-sm text-gray-600">{option.description}</div>
                    </div>
                  </button>
                );
              })}
            </div>
          </div>

          {/* Special Requirements */}
          <div>
            <Label className="block text-sm font-semibold text-gray-700 mb-3">
              Special Requirements (optional)
            </Label>
            <div className="space-y-3">
              {specialRequirements.map((requirement) => (
                <div key={requirement.id} className="flex items-center space-x-3">
                  <Checkbox
                    id={requirement.id}
                    checked={requirements.includes(requirement.id)}
                    onCheckedChange={() => toggleRequirement(requirement.id)}
                  />
                  <Label htmlFor={requirement.id} className="text-gray-700 cursor-pointer">
                    {requirement.label}
                  </Label>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>

      {/* Desktop Buttons */}
      <div className="hidden md:flex justify-between p-8 pt-0">
        <Button 
          onClick={onBack}
          variant="outline"
          className="px-6 py-3"
        >
          ← Back
        </Button>
        <Button 
          onClick={onNext}
          disabled={!isValid}
          className="bg-eco-primary text-white px-8 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
        >
          Continue to Contact Info
          <span className="ml-2">→</span>
        </Button>
      </div>

      {/* Mobile Sticky Buttons */}
      <div className="md:hidden fixed bottom-0 left-0 right-0 p-4 bg-white border-t border-gray-200 shadow-lg">
        <div className="flex gap-3">
          <Button 
            onClick={onBack}
            variant="outline"
            className="flex-1 py-4"
          >
            ← Back
          </Button>
          <Button 
            onClick={onNext}
            disabled={!isValid}
            className="flex-2 bg-eco-primary text-white py-4 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
          >
            Continue
            <span className="ml-2">→</span>
          </Button>
        </div>
      </div>
    </div>
  );
}
