import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Mail, Phone, MessageSquare, Shield } from "lucide-react";
import { WizardData } from "@/lib/types";

interface ContactInfoProps {
  data: WizardData;
  onUpdate: (data: Partial<WizardData>) => void;
  onNext: () => void;
  onBack: () => void;
  onSubmit: () => void;
}

const contactMethods = [
  {
    id: "email",
    title: "Email",
    description: "Fastest response",
    icon: Mail
  },
  {
    id: "phone",
    title: "Phone",
    description: "Direct conversation",
    icon: Phone
  },
  {
    id: "both",
    title: "Both",
    description: "Either method",
    icon: MessageSquare
  }
];

export function ContactInfo({ data, onUpdate, onNext, onBack, onSubmit }: ContactInfoProps) {
  const contactData = {
    name: data.name || "",
    email: data.email || "",
    phone: data.phone || "",
    company: data.company || "",
    preferredContact: data.preferredContact || ""
  };

  const updateField = (field: string, value: string) => {
    onUpdate({ [field]: value });
  };

  const selectContactMethod = (method: string) => {
    onUpdate({ preferredContact: method });
  };

  const isValid = contactData.name && contactData.email && contactData.phone;

  const handleSubmitAndNext = () => {
    if (isValid) {
      onSubmit(); // Submit lead to CRM and email
      onNext();   // Move to quote display
    }
  };

  return (
    <div className="flex flex-col h-full">
      <div className="flex-1 p-8 pb-24 md:pb-8">
        <div className="text-center mb-8">
          <h3 className="text-2xl font-bold text-eco-secondary mb-2">Contact information</h3>
          <p className="text-gray-600">We'll use this to send your quote and coordinate service</p>
        </div>

        <div className="max-w-2xl mx-auto space-y-6">
          {/* Contact Form */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <Label className="block text-sm font-semibold text-gray-700 mb-2">
                Full Name *
              </Label>
              <Input
                type="text"
                placeholder="John Doe"
                value={contactData.name}
                onChange={(e) => updateField("name", e.target.value)}
                required
                className="border-2 border-gray-400 focus:border-eco-primary"
                data-testid="input-name"
              />
            </div>
            <div>
              <Label className="block text-sm font-semibold text-gray-700 mb-2">
                Email Address *
              </Label>
              <Input
                type="email"
                placeholder="john@company.com"
                value={contactData.email}
                onChange={(e) => updateField("email", e.target.value)}
                required
                className="border-2 border-gray-400 focus:border-eco-primary"
                data-testid="input-email"
              />
            </div>
            <div>
              <Label className="block text-sm font-semibold text-gray-700 mb-2">
                Phone Number *
              </Label>
              <Input
                type="tel"
                placeholder="(555) 123-4567"
                value={contactData.phone}
                onChange={(e) => updateField("phone", e.target.value)}
                required
                className="border-2 border-gray-400 focus:border-eco-primary"
                data-testid="input-phone"
              />
            </div>
            <div>
              <Label className="block text-sm font-semibold text-gray-700 mb-2">
                Company (optional)
              </Label>
              <Input
                type="text"
                placeholder="Company Name"
                value={contactData.company}
                onChange={(e) => updateField("company", e.target.value)}
                className="border-2 border-gray-400 focus:border-eco-primary"
                data-testid="input-company"
              />
            </div>
          </div>

          {/* Preferred Contact Method */}
          <div>
            <Label className="block text-sm font-semibold text-gray-700 mb-3">
              Preferred Contact Method
            </Label>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
              {contactMethods.map((method) => {
                const Icon = method.icon;
                const isSelected = contactData.preferredContact === method.id;
                
                return (
                  <button
                    key={method.id}
                    onClick={() => selectContactMethod(method.id)}
                    className={`p-4 border-2 rounded-lg hover:border-eco-primary hover:bg-eco-light transition-all duration-300 ${
                      isSelected 
                        ? 'border-eco-primary bg-eco-light' 
                        : 'border-gray-200'
                    }`}
                  >
                    <div className="text-center">
                      <Icon className="text-eco-primary text-xl mb-2 w-6 h-6 mx-auto" />
                      <div className="font-semibold">{method.title}</div>
                      <div className="text-sm text-gray-600">{method.description}</div>
                    </div>
                  </button>
                );
              })}
            </div>
          </div>

          {/* Privacy Notice */}
          <div className="bg-gray-50 rounded-lg p-4">
            <div className="flex items-start space-x-3">
              <Shield className="text-eco-primary mt-1 w-5 h-5" />
              <div className="text-sm text-gray-600">
                <strong className="text-gray-900">Privacy Protection:</strong> Your information is encrypted and never shared with third parties. We use it only to provide your shredding quote and coordinate service.
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Desktop Buttons */}
      <div className="hidden md:flex justify-between p-8 pt-0">
        <Button 
          onClick={onBack}
          variant="outline"
          className="px-6 py-3"
        >
          ← Back
        </Button>
        <Button 
          onClick={handleSubmitAndNext}
          disabled={!isValid}
          className="bg-eco-primary text-white px-8 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
        >
          Get My Quote
          <span className="ml-2">→</span>
        </Button>
      </div>

      {/* Mobile Sticky Buttons */}
      <div className="md:hidden fixed bottom-0 left-0 right-0 p-4 bg-white border-t border-gray-200 shadow-lg">
        <div className="flex gap-3">
          <Button 
            onClick={onBack}
            variant="outline"
            className="flex-1 py-4"
          >
            ← Back
          </Button>
          <Button 
            onClick={handleSubmitAndNext}
            disabled={!isValid}
            className="flex-2 bg-eco-primary text-white py-4 rounded-lg font-semibold hover:bg-green-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
          >
            Get Quote
            <span className="ml-2">→</span>
          </Button>
        </div>
      </div>
    </div>
  );
}
